#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <libldiag.h>
#include <assert.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "enclosure.h"
#include "map.h"

struct enclosure *Switch;
int Nswitch;

extern char *Switchfile;
extern char *Tmpdir;

#define strcmp_sofar(S1,S2) strncmp(S1, S2, strlen(S2))

void
parse_fiber_port(char **w, int wc, struct fiberport *fp)
{
  if (strcmp(w[0], "portNumber") == 0) {
    fp->portno = atoi(w[1]);
  } else if (strcmp(w[0], "signalLost") == 0) {
    fp->vals.signallost = atoi(w[1]);
  } else if (strcmp(w[0], "signalLostCount") == 0) {
    fp->vals.signallostcount = atoi(w[1]);
  } else if (strcmp(w[0], "controlForShortwaveFiberPort") == 0) {
    fp->vals.controlforfiberport = atoi(w[1]);
  }
}

void
parse_xbar_port(char **w, int wc, struct xbarport *xpp)
{
  if (strcmp(w[0], "portNumber") == 0) {
    xpp->portno = atoi(w[1]);
  } else if (strcmp(w[0], "invalidRoutes") == 0) {
    xpp->vals.invalidroutes = atoi(w[1]);
  } else if (strcmp(w[0], "badCrcs") == 0) {
    xpp->vals.badcrcs = atoi(w[1]);
  } else if (strcmp(w[0], "goodCrcs") == 0) {
    xpp->vals.goodcrcs = atoi(w[1]);
  } else if (strcmp(w[0], "receiveTimeoutCount") == 0) {
    xpp->vals.receivetimeoutcount = atoi(w[1]);
  } else if (strcmp(w[0], "transmitTimeoutCount") == 0) {
    xpp->vals.transmittimeoutcount = atoi(w[1]);
  } else if (strcmp(w[0], "controlForXbarPort") == 0) {
    xpp->vals.controlforxbarport = atoi(w[1]);
  }
}

/*
 * Parse a line card with an xbar and 8 fiber ports
 */
int
parse_slot(
  FILE *p,
  struct enclosure *ep,
  int slotno)
{
  struct slotdesc *sp;
  char buf[256];
  char **w;
  int wc;
  int num;

  /* Get pointer to this slot */
  sp = ep->slot[slotno];

  /* get the next line */
  if (fgets(buf, sizeof(buf), p) == NULL) {
    fprintf(stderr, "Unexpected EOF in %s at %d\n", __FILE__, __LINE__);
    return -1;
  }
  w = line2words(buf, "	 ", 0, &wc);

  /* parse xbar ports */
  num = -1;
  while (1) {

    /* only deal with lines with attributes and values */
    if (wc > 1) {

      /* parse the xbar stanza */
      if (strcmp(w[0], "XbarPort") == 0) {

	/* fill in parent info */
	++num;
        if (num >= sp->num_xp) break;

	if (sp->xp[num].xbarportno != atoi(w[1])) {
	  fprintf(stderr, "%s: Xbarport mismatch: %d != %d\n",  sp->ep->name, 
	  	sp->xp[num].xbarportno, atoi(w[1]));
		return -1;
	}
	sp->xp[num].ep = ep;
	sp->xp[num].slotno = slotno;

      /* If we see start of external ports, break out */
      } else if ((sp->fiber_port_string != NULL &&
                 strcmp(w[0], sp->fiber_port_string) == 0) ||
      		  strcmp(w[0], "OperatingLed") == 0 ||
      		  strcmp(w[0], "Voltage") == 0) {

	/* keep w for next loop to use */
	break;

      } else if (num >= 0) {

	/* parse the port info */
	parse_xbar_port(w, wc, &(sp->xp[num]));
      }
    }

    free(w);
    w = NULL;

    /* get the next line */
    if (fgets(buf, sizeof(buf), p) != NULL) {
      w = line2words(buf, "	 ", 0, &wc);
    } else {
      if (num != (sp->num_xp - 1)) {
	fprintf(stderr, "bad number of xbars in file[1]\n");
	return -1;
      }
      goto done_with_slot;
    }
  }

  /* sanity check number of Xbars */
  if (num != (sp->num_xp - 1)) {
    free(w);
    fprintf(stderr, "bad num xbars [2] ");
    fprintf(stderr, "num=%d, num_xp=%d ", num, sp->num_xp);
    fprintf(stderr, "enc %s/%d [%p]\n", ep->name, slotno, w);
    return -1;
  }

  assert(w != NULL);

  /* Parse fiber ports, if there are any */
  if (sp->num_fiber > 0) {

    /* parse fiber port info */
    num = -1;
    while (1) {

      /* parse the xbar stanza */
      if (wc > 1) {
	if (strcmp(w[0], sp->fiber_port_string) == 0) {

	  /* fill in parent info */
	  ++num;
          if (num >= sp->num_fiber) {
	    free(w);
	    w = NULL;
	    break;
	  }
	  sp->fp[num].ep = ep;
	  sp->fp[num].slotno = slotno;

	} else if (strcmp(w[0], "Voltage") == 0) {

	  /* all done with slot */
	  free(w);
          w = NULL;
	  break;

	} else if (num >= 0) {

	  /* parse the fiber port info */
	  parse_fiber_port(w, wc, &(sp->fp[num]));
	}
      }

      free(w);
      w = NULL;

      /* get the next line */
      if (fgets(buf, sizeof(buf), p) != NULL) {
	w = line2words(buf, "	 ", 0, &wc);
      } else {
	break;
      }
    }
    if (num != (sp->num_fiber - 1)) {
      fprintf(stderr, "%s: bad number of fiber ports in file\n", ep->name);
      assert(w == NULL);
      return -1;
    }

  } else {
    free(w);
    w = NULL;
  }

 done_with_slot:

  /* Fill in label offset based on what we know about the card */
  if (sp->num_xp == 16) {
    sp->port_label_offset = 8;
  } else {
    sp->port_label_offset = 0;
  }
  assert(w == NULL);

  return 0;
}

void
query_slot(struct enclosure *ep, int slot)
{
  FILE *p;
  char cmd[256];
  int tries;
  int rc;

  tries = 0;
  do {
    /* generate command line to query the swtch slot */
    sprintf(cmd, "wget -q -O - http://%s/slot%d | sed 's/<[^>]*>//g'",
	  ep->name, slot);

    p = popen(cmd, "r");
    if (p == NULL) {
      perror("popen");
      exit(1);
    }

    /* parse the line card */
    rc = parse_slot(p, ep, slot);

    pclose(p);

  } while (rc != 0 && ++tries <= 3);

  assert(rc == 0);
}

/*
 * allocate and fill in a slot descriptor
 */
int
fill_slot_desc(
  FILE *p,
  struct enclosure *ep,
  int slotno,
  int num_xbar_port,
  int num_fiber_port,
  char *fiber_port_string)
{
  struct slotdesc *sp;
  int sl;
  int port;
  int xbarportno;

  /* pointer to slot descriptor and back */
  sp = ep->slot[slotno];
  sp->ep = ep;
  sp->slotno = slotno;

  /* Get initial XbarPort */
  xbarportno = 1;
  for (sl=1; sl<slotno; ++sl) {
    if (ep->slot[sl] != NULL) {
      xbarportno += ep->slot[sl]->num_xp;
    }
  }
  
  /* record how many of each port type */
  sp->num_xp = num_xbar_port;
  sp->num_fiber = num_fiber_port;
  sp->fiber_port_string = fiber_port_string;

  /* allocate space for xbar ports, if needed */
  if (sp->xp == NULL && num_xbar_port != 0) {
    sp->xp = (struct xbarport *) calloc(sizeof(struct xbarport), sp->num_xp);
    assert(sp->xp != NULL);
  }

  /* fill in xbarportno */
  for (port=0; port<num_xbar_port; ++port) {
    sp->xp[port].xbarportno = xbarportno;
    ++xbarportno;
  }

  if (sp->fp == NULL && sp->num_fiber != 0) {
    sp->fp = (struct fiberport *)
	calloc(sizeof(struct fiberport), sp->num_fiber);
    assert(sp->fp != NULL);
  }

  /* parse input stream into the structs */
  return parse_slot(p, ep, slotno);
}

/*
 * Link the fiber ports of any lines card to the xbar on the same card
 */
void
link_line_cards_to_self(struct enclosure *ep)
{
  int slot;
  int p;
  struct slotdesc *sp;
  struct xbarport *xp;
  struct fiberport *fp;

  for (slot=1; slot<MAX_SLOT; ++slot) {
    sp = ep->slot[slot];
    if (sp == NULL || sp->num_xp == 0 || sp->num_fiber == 0) continue;

    /* link each fiber port to an xbar port */
    for (p=0; p<sp->num_fiber; ++p) {
      fp = sp->fp + p;
      xp = sp->xp + p + 8;
      fp->xp = xp;
      xp->fp = fp;
    }
  }
}


/*
 * Assume one line card and one spine for 16, error otherwise
 */
void
make_static_links_16(struct enclosure *ep)
{
  int slot;
  int p;
  struct slotdesc *sp;
  struct slotdesc *sp2;
  struct xbarport *xp;
  struct fiberport *fp;

  /* link fiber ports for any line cards */
  link_line_cards_to_self(ep);

  /* Now, handle a spine card if there is one */
  for (slot=1; slot<=2; ++slot) {
    sp = ep->slot[slot];
    if (sp == NULL || sp->num_xp > 0) continue;

    /* make sure the other slot has an xbar */
    sp2 = ep->slot[3-slot];
    if (sp2 == NULL || sp2->num_xp == 0) continue;

    /* OK, link our fiber ports to hix xbars */
    for (p=0; p<8; ++p) {
      fp = sp->fp + p;
      xp = sp2->xp + p;
      fp->xp = xp;
      xp->fp = fp;
    }
  }
}

void
link_cards_to_backplane(struct enclosure *ep)
{
  struct slotdesc *sp;
  struct slotdesc *bpsp;
  int bpport;
  int bpport_skip;
  int bpslot;
  int port;
  int slot;

  for (slot = 1; slot <= 16; ++slot) {
    sp = ep->slot[slot];

    /* skip non-existant cards */
    if (sp == NULL) continue;

    bpport_skip = ep->num_bp_slots * 2;
    bpslot = 33;
    bpport = slot-1;

    for (port=0; port<8; ++port) {
      bpsp = ep->slot[bpslot];

      /* if card has xbar, link the two xbarport structs together */
      if (sp->num_xp > 0) {
	sp->xp[port].xp = bpsp->xp + bpport;
	bpsp->xp[bpport].xp = sp->xp + port;


      /* else, just link to the fiber port */
      } else {
        sp->fp[port].xp = bpsp->xp + bpport;
	bpsp->xp[bpport].fp = sp->fp + port;
      }

      /* update port and slot for next iteration */
      bpport += bpport_skip;
      if (bpport > 15) {
	++bpslot;
	bpport = slot-1;
      }
    }
  }
}

void
make_static_links_big(struct enclosure *ep)
{
  /* link fiber ports for any line cards */
  link_line_cards_to_self(ep);

  /* Now, link all the cards to the backplane xbars */
  link_cards_to_backplane(ep);
}

/*
 * query an entire enclosure
 */
int
parse_enclosure_stream(
  FILE *fp,
  struct enclosure *ep)
{
  char buf[256];
  char **w;
  int wc;
  int slotno;
  struct slotdesc *sp;
  int rc;

  // printf("parsing %s\n", ep->name);

  /* clear port counts */
  ep->num_ext_slots = 0;
  ep->num_bp_slots = 0;

  while (fgets(buf, sizeof(buf), fp) != NULL) {
    w = line2words(buf, "	 \n", 0, &wc);

    /* looking for "Slot N type" */
    if (wc < 3) {
      free(w);
      continue;
    }

    if (strcmp(w[0], "Slot") == 0) {

      /* slot number this describes */
      slotno = atoi(w[1]);

      /* skip big slots */
      if (slotno >= MAX_SLOT) {
	free(w);
	continue;
      }

      /* get pointer to descriptor for the slot */
      sp = ep->slot[slotno];
      if (sp == NULL) {
	sp = (struct slotdesc *) calloc(sizeof(struct slotdesc), 1);
	assert(sp != NULL);
	ep->slot[slotno] = sp;	/* save pointer in enclosure struct */
      }

      /* Fill in slot type */
      if (w[2][0] == '(') {
	strcpy(sp->type, w[2]+1);	/* skip leading paren */
	if (sp->type[strlen(sp->type)-1] == ')') {
	  sp->type[strlen(sp->type)-1] = '\0';	/* knock off trailing paren */
	}
      } else {
	assert(0);	/* require type for now */
	strcpy(sp->type, "UNKNOWN");
      }
      free(w);

      /* Get model string */
      while (fgets(buf, sizeof(buf), fp) != NULL) {
	w = line2words(buf, "	 \n", 0, &wc);
	if (wc > 0 && strcmp(w[0], "serialNumber") == 0) {
	  if (wc > 2) {
	    strcpy(sp->serialno, w[1]);
	    strcpy(sp->model, w[2]);
	  } else {
	    strcpy(sp->serialno, "n/a");
	    strcpy(sp->model, "UNKNOWN");
	  }
	  free(w);
	  break;
	} else {
	  free(w);
	}
      }
        
      if (strcmp(sp->type, "backplane") == 0) {
        rc = fill_slot_desc(fp, ep, slotno, 16, 0, NULL);
        ++ep->num_bp_slots;

      } else if (strcmp(sp->type, "lc8sx") == 0) {
        rc = fill_slot_desc(fp, ep, slotno, 16, 8, "HssdcPort");
        ++ep->num_ext_slots;

      } else if (strcmp(sp->model, "M3-SW16-8F") == 0) {
        rc = fill_slot_desc(fp, ep, slotno, 16, 8, "ShortwaveFiberPort");
        ++ep->num_ext_slots;

      } else if (strcmp(sp->type, "lc8s") == 0) {
        rc = fill_slot_desc(fp, ep, slotno, 0, 8, "HssdcPort");
        ++ep->num_ext_slots;

      } else if (strcmp(sp->type, "lc8fx") == 0) {
        rc = fill_slot_desc(fp, ep, slotno, 16, 8, "ShortwaveFiberPort");
        ++ep->num_ext_slots;

      } else if (strcmp(sp->type, "lc8fx2") == 0) {
        rc = fill_slot_desc(fp, ep, slotno, 16, 8, "ShortwaveFiberPort");
        ++ep->num_ext_slots;

      } else if (strcmp(sp->type, "lc8f2x") == 0) {
        rc = fill_slot_desc(fp, ep, slotno, 16, 8, "ShortwaveFiberPort");
        ++ep->num_ext_slots;

      } else if (strcmp(sp->type, "lc8f2") == 0) {
        rc = fill_slot_desc(fp, ep, slotno, 0, 8, "ShortwaveFiberPort");
        ++ep->num_ext_slots;

      } else if (strcmp(sp->type, "lc8ex") == 0) {
        rc = fill_slot_desc(fp, ep, slotno, 16, 8, "GEPort");
        ++ep->num_ext_slots;

      } else if (strcmp(sp->type, "lc4dm") == 0) {
        rc = fill_slot_desc(fp, ep, slotno, 0, 4, "SANPort");
        ++ep->num_ext_slots;

      } else {
        fprintf(stderr, "Unknown slot type \"%s\" in %s\n", sp->type, ep->name);
	return -1;
      }

      /* propagate any error */
      if (rc != 0) {
        return rc;
      }

    } else {
      free(w);
    }
  }

  /* Do a little sanity check */
  if (ep->num_bp_slots != 0 &&
      ep->num_bp_slots != 2 &&
      ep->num_bp_slots != 4 &&
      ep->num_bp_slots != 8) {
    fprintf(stderr, "bad backplane slot count=%d %s\n",
            ep->num_bp_slots, ep->name);
    return -1;
  }

  return 0;
}

/*
 * query an entire enclosure
 */
int
parse_enclosure_from_file(
  struct enclosure *ep)
{
  FILE *p;
  char fname[256];
  int rc;

  /* generate command line to query the swtch slot */
  sprintf(fname, "%s/%s", Tmpdir, ep->name);

  p = fopen(fname, "r");
  if (p == NULL) {
    perror(fname);
    exit(1);
  }

  /* parse the indo */
  rc = parse_enclosure_stream(p, ep);

  fclose(p);

  return rc;
}

void
make_static_links()
{
  int s;
  struct enclosure *ep;

  for (s=0; s<Nswitch; ++s) {
    ep = Switch + s;

    /* Make static links */
    switch (ep->num_bp_slots) {
    case 0:
      make_static_links_16(ep);
      break;
    case 2:
      make_static_links_big(ep);
      break;
    case 4:
      make_static_links_big(ep);
      break;
    case 8:
      make_static_links_big(ep);
      break;
    default:
      assert(0);
      break;
    }
  }
}

void
query_enclosure(
  struct enclosure *ep)
{
  FILE *p;
  char cmd[256];
  int rc;
  int tries;

  tries = 0;
  do {

    sprintf(cmd, "wget -q -O - http://%s/all | sed 's/<[^>]*>//g'", ep->name);
    p = popen(cmd, "r");
    if (p == NULL) {
      perror("popen");
      exit(1);
    }

    rc = parse_enclosure_stream(p, ep);

    pclose(p);

  } while (rc != 0 && tries <= 3);

  assert(rc == 0);
}

void
parse_all_enclosures()
{
  int s;
  char cmd[16000];
  int num_valid;
  int rc;
  int status;
  int tries;

  /* mark all switches as invalid, then loop until all are valid */
  for (s=0; s<Nswitch; ++s) {
    int slot;
    int num;
    struct slotdesc *sp;

    Switch[s].data_valid = 0;

    /* save all old data values for each line card */
    for (slot=0; slot<MAX_SLOT; ++slot) {
      sp = Switch[s].slot[slot];

      if (sp != NULL) {

	/* save old values for each xbar port */
	for (num=0; num < sp->num_xp; ++num) {
	  sp->xp[num].old_vals = sp->xp[num].vals;
	}


	/* save old values for each fiber port */
	for (num=0; num < sp->num_fiber; ++num) {
	  sp->fp[num].old_vals = sp->fp[num].vals;
	}
      }
    }
  }
  num_valid = 0;

  /* wait until all switches have valid data */
  tries = 0;
  while (num_valid < Nswitch && ++tries <= 50) {

    printf("starting parallel query...\n");
    for (s=0; s<Nswitch; ++s) {

      if (Switch[s].data_valid) {	/* If already valid, skip */
        continue;
      }

      Switch[s].pid = fork();

      /* The child process will create the file */
      if (Switch[s].pid == 0) {
	int off;

	off = sprintf(cmd, "( ");
	off += sprintf(cmd+off, "wget -q -O - http://%s/all", Switch[s].name);
	off += sprintf(cmd+off, ") | sed 's/<[^>]*>//g' > %s/%s",
	    			Tmpdir, Switch[s].name);

	if (system(cmd) != 0) {
	  fprintf(stderr, "Error executing %s\n", cmd);
	  exit(1);
	}

	exit(0);
      }
    }

    /* Wait for all children to finish */
    for (s=0; s<Nswitch; ++s) {

      if (Switch[s].data_valid) {	/* If already valid, skip */
        continue;
      }

      rc = waitpid(Switch[s].pid, &status, 0);
      if (rc == -1) {
	perror("waitpid");
	exit(1);
      }
      if (status != 0) {
	printf("Child handling %s had a problem\n", Switch[s].name);
	exit(1);
      }

      /* Parse the switch file */
      if (parse_enclosure_from_file(Switch+s) == 0) {
        Switch[s].data_valid = 1;
	++num_valid;
      }

    }

    printf("%d/%d switches valid\n", num_valid, Nswitch);
  }
  assert(num_valid == Nswitch);
  printf("All switches queried!\n");
}


/*
 * Read in the switch file and create an enclosure struct for each switch
 */
void
load_switch_file()
{
  FILE *fp;
  char buf[256];
  int ns;
  int slot;
  struct enclosure *ep;
  struct slotdesc *sp;

  Nswitch = 0;

  fp = fopen(Switchfile, "r");
  assert(fp != NULL);
  while (fgets(buf, sizeof(buf), fp) != NULL) {
    if (buf[0] == '\n' || buf[0] == '#') {
      continue;
    }
    ++Nswitch;
  }
  fclose(fp);

  printf("%d switch%s\n", Nswitch, Nswitch>1?"es":"");
  Switch = (struct enclosure *) calloc(sizeof(struct enclosure), Nswitch);
  assert(Switch != 0);

  ns = 0;

  fp = fopen(Switchfile, "r");
  assert(fp != NULL);
  while (fgets(buf, sizeof(buf), fp) != NULL) {
    if (buf[0] == '\n' || buf[0] == '#') {
      continue;
    }
    
    buf[strlen(buf)-1] = '\0';	/* remove newline */
    strcpy(Switch[ns].name, buf);

    ns++;
  }
  fclose(fp);

  /* get a baseline */
  parse_all_enclosures();

  /* Fill in what we know about the internal topology */
  make_static_links();

  /* remember how many slots we need to map */
  for (ns=0; ns<Nswitch; ++ns) {
    ep = Switch + ns;
    ep->unmapped_slots = 0;
    for (slot=1; slot<MAX_SLOT; ++slot) {
      sp = ep->slot[slot];
      if (sp != NULL && sp->num_xp > 0) {
        ++ep->unmapped_slots;
      }
    }
  }

}

/*
 * Propagate information out from a card with an xbar
 */
void
propagate_from(struct slotdesc *sp)
{
  struct slotdesc *sp2;
  struct xbar *xbar;
  struct xbar *xbar2;
  struct xbarport *xp;
  int xport;
  int wport;

  xbar = sp->xbar;
  for (xport=0; xport<16; ++xport) {

    /* get pointer to wiring xbar struct */
    wport = XPORT_TO_WPORT(xport, sp->port_delta);
    xp = sp->xp + wport;

    /* for each connection to an xbar */
    if (xbar->port[xport].conn_type == CONN_XBAR) {
      /* if no xp pointer in xp, then this goes to another enclosure */
      if (xp->xp == NULL) {
        assert(xp->fp != NULL);
	xp->fp->plug_type = PLUG_TYPE_PLUG;
        continue;
      }

      /* get pointer to the connected xbar */
      xbar2 = xbar->port[xport].ptr.x;

      /* slot holding other connected xbarport */
      sp2 = xp->xp->ep->slot[xp->xp->slotno];

      /* if already known, skip it */
      if (sp2->xbar != NULL) {
        assert(sp2->xbar == xbar2);
	assert(xbar2->sp == sp2);
        continue;
      }

      /* Make the connection */
      xbar2->sp = sp2;
      sp2->xbar = xbar2;
      sp2->port_delta =
      	MAKE_PORT_DELTA(xp->xp->portno, xbar->port[xport].conn_port);

      {
        struct xbarport *xp2;
     
        xp2 = sp2->xp + XPORT_TO_WPORT(xbar->port[xport].conn_port,
				sp2->port_delta);
        assert(xp2->xp == xp);
        assert(xp->xp == xp2);
      }

      /* Account for mapping this slot */
      assert(sp->ep->unmapped_slots > 0);
      --sp->ep->unmapped_slots;

      propagate_from(sp2);
    }
  }
}

/*
 * bind a port on this backplane card to one known xbar
 */
void
associate_backplane_card(
  struct enclosure *ep,
  int slot,
  int wport,
  struct xbar *xbar,
  int xport)
{

printf("associating to backplane %d, wp=%d, xp=%d\n", slot, wport, xport);
}
/*
 * bind a port on this line card to one known xbar
 */
void
associate_line_card(
  struct enclosure *ep,
  int slot,
  int wport,
  struct xbar *xbar,
  int xport)
{
  struct slotdesc *sp;

  /* pointer for this slot descriptor */
  sp = ep->slot[slot];

  /* If already mapped, skip it */
  if (sp->xbar != NULL) {
    assert(sp->xbar == xbar);
    assert(xbar->sp == sp);
    return;
  }

  /* link this physical xbar to xbar in map */
  sp->xbar = xbar;
  sp->port_delta = MAKE_PORT_DELTA(wport, xport);
  xbar->sp = sp;

  /* Account for mapping this slot */
  assert(ep->unmapped_slots > 0);
  --ep->unmapped_slots;

  propagate_from(sp);
}

/*
 * bind this enclosure given one known xbar
 */
void
associate_enclosure(
  struct enclosure *ep,
  int slot,
  int wport,
  struct xbar *xbar,
  int xport)
{
  struct slotdesc *sp;

  /* pointer for this slot descriptor */
  sp = ep->slot[slot];
  assert (sp->num_xp > 0);

  /* treat line card and backplane card differently */
  if (1 || sp->num_fiber > 0) {
    associate_line_card(ep, slot, wport, xbar, xport);
  } else {
    associate_backplane_card(ep, slot, wport, xbar, xport);
  }
}
